//====================================================================================================
// 27/11/2024 : V1.4: Correction affichage dynamique des unités (W/mW/µW/nW)
// 29/09/2025 : V1.3: version stable - Développement avec DeepSeek
//====================================================================================================
#include <Adafruit_GFX.h>
#include <Adafruit_ILI9341.h>
#include <EEPROM.h>
#include "RF_PowerMeter.h"

// Broches SPI et codeurs - ARDUINO NANO
#define TFT_MOSI  11
#define TFT_SCK   13
#define TFT_CS    10
#define TFT_DC    9
#define TFT_RST   8
#define FREQ_ATT_ENCODER_A   2
#define FREQ_ATT_ENCODER_B   4
#define MENU_ENCODER_A       3
#define MENU_ENCODER_B       5
#define PUSH_BUTTON          6

// Adresses EEPROM
#define EEPROM_MAGIC_ADDR     0
#define EEPROM_FREQ_ADDR      2
#define EEPROM_ATT1_ADDR      6
#define EEPROM_ATT2_ADDR      10
#define EEPROM_STEP_ADDR      14
#define EEPROM_UNITS_ADDR     15
#define EEPROM_MAGIC_VALUE    0xAB42

// Couleurs
#define COLOR_BG ILI9341_BLACK
#define COLOR_HDR ILI9341_CYAN
#define COLOR_VAL ILI9341_YELLOW
#define COLOR_CH1 ILI9341_GREEN

Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC, TFT_MOSI, TFT_SCK, TFT_RST);

#define Vref 3.33
RF_PowerMeter powerMeter(A1, Vref);
RF_PowerMeter powerMeter2(A0, Vref);

// Variables principales
bool displayLinearUnits = true;
unsigned long lastDisplayTime = 0;
const unsigned long DISPLAY_INTERVAL = 250;
unsigned long lastSaveTime = 0;
const unsigned long SAVE_DELAY = 2000;

// Variables de sauvegarde
bool needSave = false;

// Variables codeurs
volatile int valueEncoderPos = 0;
volatile bool valueEncoderA_prev;
volatile int menuEncoderPos = 0;
volatile bool menuEncoderA_prev;

// Variables contrôle
float currentFreq = 145.0;
float freqSteps[] = {0.1, 1.0, 10.0, 100.0};
int currentStepIndex = 1;
float currentAtt1 = 0.0;
float currentAtt2 = 0.0;

enum ControlMode { MODE_FREQUENCY, MODE_FREQ_STEP, MODE_ATT_SONDE1, MODE_ATT_SONDE2 };
ControlMode currentMode = MODE_FREQUENCY;

// Variables affichage
float lastPower1 = -999.9, lastPower2 = -999.9;
float lastFreq = -1.0, lastAtt = -1.0, lastAtt2 = -1.0, lastStep = -1.0;
float lastPowerWatts1 = -999.9, lastPowerWatts2 = -999.9;
float lastGainDb = -999.9;
String lastMessage = "";
String lastUnit1 = "", lastUnit2 = "";  // Mémoriser les unités affichées

bool controlParamsNeedUpdate = true;

// ISR pour encodeur réglage des valeurs
void valueEncoderISR() {
  bool a = digitalRead(FREQ_ATT_ENCODER_A);
  if (valueEncoderA_prev != a) {
    valueEncoderPos += (a == digitalRead(FREQ_ATT_ENCODER_B)) ? -1 : 1;
  }
  valueEncoderA_prev = a;
}

// ISR pour encodeur sélection menu
void menuEncoderISR() {
  bool a = digitalRead(MENU_ENCODER_A);
  if (menuEncoderA_prev != a) {
    menuEncoderPos += (a == digitalRead(MENU_ENCODER_B)) ? -1 : 1;
  }
  menuEncoderA_prev = a;
}

void setup() {
  Serial.begin(9600);
  
  analogReference(EXTERNAL);
  powerMeter.setVref(Vref);
  powerMeter2.setVref(Vref);
  powerMeter.setSonde(SONDE_1);
  powerMeter2.setSonde(SONDE_2);
  
  // Configuration broches
  pinMode(TFT_MOSI, OUTPUT); 
  pinMode(TFT_SCK, OUTPUT); 
  pinMode(TFT_CS, OUTPUT); 
  pinMode(TFT_DC, OUTPUT); 
  pinMode(TFT_RST, OUTPUT);
  pinMode(FREQ_ATT_ENCODER_A, INPUT_PULLUP); 
  pinMode(FREQ_ATT_ENCODER_B, INPUT_PULLUP);
  pinMode(MENU_ENCODER_A, INPUT_PULLUP); 
  pinMode(MENU_ENCODER_B, INPUT_PULLUP);
  pinMode(PUSH_BUTTON, INPUT_PULLUP);
  
  // Init écran
  digitalWrite(TFT_RST, HIGH); 
  delay(10); 
  digitalWrite(TFT_RST, LOW); 
  delay(10);
  digitalWrite(TFT_RST, HIGH); 
  delay(150);
  tft.begin(8000000);
  tft.setRotation(3);

  // Init encodeurs
  valueEncoderA_prev = digitalRead(FREQ_ATT_ENCODER_A);
  menuEncoderA_prev = digitalRead(MENU_ENCODER_A);
  attachInterrupt(0, valueEncoderISR, CHANGE);
  attachInterrupt(1, menuEncoderISR, CHANGE);
  
  // Chargement des paramètres depuis l'EEPROM
  loadFromEEPROM();
  
  powerMeter.setFrequency(currentFreq);
  powerMeter2.setFrequency(currentFreq);
  powerMeter.setAttenuation(currentAtt1);
  powerMeter2.setAttenuation(currentAtt2);
  
  drawUI();
}

void loop() {
  handleMenuEncoder();
  handleValueEncoder();
  handleSerial();

  if (needSave && (millis() - lastSaveTime > SAVE_DELAY)) {
    saveToEEPROM();
    needSave = false;
  }

  if (millis() - lastDisplayTime >= DISPLAY_INTERVAL) {
    lastDisplayTime = millis();
    
    // LECTURE DES MESURES DE PUISSANCE
    float voltage = powerMeter.readVoltage();
    float power   = powerMeter.calculateCorrectedPower();
    float powerWatts = displayLinearUnits ? powerMeter.convertToWatts(power) : 0;
    
    float voltage2 = powerMeter2.readVoltage();
    float power2   = powerMeter2.calculateCorrectedPower();
    float powerWatts2 = displayLinearUnits ? powerMeter2.convertToWatts(power2) : 0;
    
    updateDisplay(power, powerWatts, power2, powerWatts2);
  }
}

void handleMenuEncoder() {
  static int lastMenuEncoderPos = 0;
  
  if (menuEncoderPos != lastMenuEncoderPos) {
    int diff = menuEncoderPos - lastMenuEncoderPos;
    lastMenuEncoderPos = menuEncoderPos;
    
    int newMode = (int)currentMode + diff;
    if (newMode < 0) newMode = 3;
    if (newMode > 3) newMode = 0;
    currentMode = (ControlMode)newMode;
    
    lastFreq = -1.0;
    lastAtt = -1.0; 
    lastAtt2 = -1.0;
    lastStep = -1.0;
    
    controlParamsNeedUpdate = true;
    updateAllDisplay();
    
    const char* modes[] = {"FREQUENCE", "PAS FREQ", "ATT VOIE1", "ATT VOIE2"};
    updateMessage(modes[currentMode]);
  }
}

void handleValueEncoder() {
  static int lastValueEncoderPos = 0;
  
  if (valueEncoderPos != lastValueEncoderPos) {
    int diff = valueEncoderPos - lastValueEncoderPos;
    lastValueEncoderPos = valueEncoderPos;
    
    switch (currentMode) {
      case MODE_FREQUENCY:
        currentFreq += diff * freqSteps[currentStepIndex];
        if (currentFreq < 1.0) currentFreq = 1.0;
        if (currentFreq > 1000.0) currentFreq = 1000.0;
        powerMeter.setFrequency(currentFreq);
        powerMeter2.setFrequency(currentFreq);
        controlParamsNeedUpdate = true;
        updateFreqDisplay();
        needSave = true;
        lastSaveTime = millis();
        break;
        
      case MODE_FREQ_STEP:
        currentStepIndex += diff;
        if (currentStepIndex < 0) currentStepIndex = 0;
        if (currentStepIndex > 3) currentStepIndex = 3;
        controlParamsNeedUpdate = true;
        updateStepDisplay();
        needSave = true;
        lastSaveTime = millis();
        break;
        
      case MODE_ATT_SONDE1:
        currentAtt1 += diff * 0.1;
        if (currentAtt1 < 0.0) currentAtt1 = 0.0;
        if (currentAtt1 > 60.0) currentAtt1 = 60.0;
        powerMeter.setAttenuation(currentAtt1);
        controlParamsNeedUpdate = true;
        updateAttDisplay();
        needSave = true;
        lastSaveTime = millis();
        break;
        
      case MODE_ATT_SONDE2:
        currentAtt2 += diff * 0.1;
        if (currentAtt2 < 0.0) currentAtt2 = 0.0;
        if (currentAtt2 > 60.0) currentAtt2 = 60.0;
        powerMeter2.setAttenuation(currentAtt2);
        controlParamsNeedUpdate = true;
        updateAttDisplay();
        needSave = true;
        lastSaveTime = millis();
        break;
    }
  }
}

void handleSerial() {
  if (Serial.available()) {
    char cmd = Serial.read();
    if (cmd == 'h' || cmd == 'H') {
      Serial.println(F("=== COMMANDES ==="));
      Serial.println(F("H:Aide"));
      Serial.print(F("Freq: ")); Serial.println(currentFreq, 1);
      Serial.print(F("Step: ")); Serial.println(freqSteps[currentStepIndex], 1);
      Serial.print(F("Att1: ")); Serial.println(currentAtt1, 1);
      Serial.print(F("Att2: ")); Serial.println(currentAtt2, 1);
    }
  }
}

void saveToEEPROM() {
  EEPROM.write(EEPROM_MAGIC_ADDR, EEPROM_MAGIC_VALUE & 0xFF);
  EEPROM.write(EEPROM_MAGIC_ADDR + 1, (EEPROM_MAGIC_VALUE >> 8) & 0xFF);
  
  writeFloatEEPROM(EEPROM_FREQ_ADDR, currentFreq);
  writeFloatEEPROM(EEPROM_ATT1_ADDR, currentAtt1);
  writeFloatEEPROM(EEPROM_ATT2_ADDR, currentAtt2);
  EEPROM.write(EEPROM_STEP_ADDR, currentStepIndex);
  EEPROM.write(EEPROM_UNITS_ADDR, displayLinearUnits ? 1 : 0);
}

void loadFromEEPROM() {
  if (isEEPROMValid()) {
    currentFreq = readFloatEEPROM(EEPROM_FREQ_ADDR);
    currentAtt1 = readFloatEEPROM(EEPROM_ATT1_ADDR);
    currentAtt2 = readFloatEEPROM(EEPROM_ATT2_ADDR);
    currentStepIndex = EEPROM.read(EEPROM_STEP_ADDR);
    displayLinearUnits = EEPROM.read(EEPROM_UNITS_ADDR) != 0;
    
    if (currentFreq < 10.0 || currentFreq > 1000.0) currentFreq = 145.0;
    if (currentAtt1 < 0.0 || currentAtt1 > 60.0) currentAtt1 = 0.0;
    if (currentAtt2 < 0.0 || currentAtt2 > 60.0) currentAtt2 = 0.0;
    if (currentStepIndex < 0 || currentStepIndex > 3) currentStepIndex = 1;
  }
}

bool isEEPROMValid() {
  uint16_t magic = EEPROM.read(EEPROM_MAGIC_ADDR) | (EEPROM.read(EEPROM_MAGIC_ADDR + 1) << 8);
  return (magic == EEPROM_MAGIC_VALUE);
}

void writeFloatEEPROM(int addr, float value) {
  byte* p = (byte*)(void*)&value;
  for (int i = 0; i < sizeof(float); i++) EEPROM.write(addr++, *p++);
}

float readFloatEEPROM(int addr) {
  float value = 0.0;
  byte* p = (byte*)(void*)&value;
  for (int i = 0; i < sizeof(float); i++) *p++ = EEPROM.read(addr++);
  return value;
}

void drawUI() {
  tft.fillScreen(COLOR_BG);
  
  // Titre principal
  tft.setTextSize(1);
  tft.setTextColor(COLOR_VAL);
  tft.setCursor(70, 5);
  tft.print(F("MilliWattmetre 1MHz-1GHz F4GSC"));
  
  // Section fréquence et pas
  tft.drawRect(5, 25, 310, 35, COLOR_HDR);
  tft.setTextColor(COLOR_HDR);
  tft.setTextSize(2);
  tft.setCursor(10, 35); tft.print(F("F(MHz)="));
  tft.setTextSize(1);
  tft.setCursor(220, 40);tft.print(F("Pas(MHz)="));
  tft.setTextSize(2);

  // Section des deux voies - LABELS AVEC ESPACES POUR UNITÉS DYNAMIQUES
  tft.drawRect(5, 70, 150, 125, COLOR_CH1);
  tft.setTextColor(COLOR_CH1);
  tft.setCursor(48, 80); tft.print(F("Voie 1"));
  tft.setCursor(10, 100); tft.print(F("P(  )")); // Espace pour unité dynamique
  tft.setCursor(10, 125); tft.print(F("P(dBm)"));
  tft.setCursor(10, 150); tft.print(F("A(dB) "));

  tft.drawRect(165, 70, 150, 125, COLOR_HDR);
  tft.setTextColor(COLOR_HDR);
  tft.setCursor(208, 80); tft.print(F("Voie 2"));
  tft.setCursor(170, 100); tft.print(F("P(  )")); // Espace pour unité dynamique
  tft.setCursor(170, 125); tft.print(F("P(dBm)"));
  tft.setCursor(170, 150); tft.print(F("A(dB)"));

  // Section message - Éléments CONSTANTS
  tft.drawRect(5, 200, 310, 30, COLOR_HDR);
  tft.setTextSize(1);
  tft.setTextColor(COLOR_HDR);

  // Label "Reglage -->" CONSTANT (jamais modifié)
  tft.setCursor(10, 210); 
  tft.print(F("Reglage -->"));

  // Label "G(dB)" CONSTANT (jamais modifié)
  tft.setTextSize(2);  
  tft.setTextColor(COLOR_VAL);
  tft.setCursor(170, 208); 
  tft.print(F("G(dB)"));

  // Afficher le mode initial (cette partie sera variable)
  tft.setTextSize(1);
  tft.setTextColor(COLOR_HDR);
  tft.fillRect(82, 210, 54, 8, COLOR_BG );
  tft.setCursor(82, 210);
  tft.print(F("FREQUENCE"));

  // Reset variables
  lastPower1 = lastPower2 = -999.9;
  lastPowerWatts1 = lastPowerWatts2 = -999.9;
  lastFreq = lastAtt = lastAtt2 = lastStep = -1.0;
  lastGainDb = -999.9;
  lastMessage = "FREQUENCE";
  lastUnit1 = "";
  lastUnit2 = "";

  updateAllDisplay();
}

void updateDisplay(float power1, float powerWatts1, float power2, float powerWatts2) {
  // TOUJOURS rafraîchir les mesures de puissance
  updatePowerMeasurements(power1, powerWatts1, power2, powerWatts2);
  
  // Rafraîchir les paramètres de contrôle seulement si nécessaire
  if (controlParamsNeedUpdate) {
    updateControlParameters();
    controlParamsNeedUpdate = false;
  }
  
  // Rafraîchir UNIQUEMENT la valeur du gain (pas le label)
  updateGainValue(power1, power2);
}

void updatePowerMeasurements(float power1, float powerWatts1, float power2, float powerWatts2) {
  // dBm Voie 1 - Rafraîchi seulement si changement
  if (abs(power1 - lastPower1) > 0.1) {
    tft.fillRect(85, 125, 69, 15, COLOR_BG);
    tft.setTextSize(2);
    tft.setTextColor(COLOR_CH1);

    if (power1>=0) {
      if (power1>10.0) tft.setCursor(88, 125);
      else tft.setCursor(98, 125);
      tft.print("+");
    } else {
      if (power1>-10.0) tft.setCursor(98, 125);
      else tft.setCursor(88, 125);
    }
    if (power1 > -80) tft.print(power1, 1);
    else tft.print(F("---.--"));
    lastPower1 = power1;
  }
  
  // dBm Voie 2 - Rafraîchi seulement si changement
  if (abs(power2 - lastPower2) > 0.1) {
    tft.fillRect(246, 125, 68, 15, COLOR_BG);
    tft.setTextSize(2);
    tft.setTextColor(COLOR_HDR);
    if (power2>=0) {
      if (power2>10.0) tft.setCursor(246, 125);
      else tft.setCursor(258, 125);
      tft.print("+");
    } else {
      if (power2>-10.0) tft.setCursor(258, 125);
      else tft.setCursor(246, 125);
    }
    if (power2 > -80) tft.print(power2, 1);
    else tft.print(F("---.--"));
    lastPower2 = power2;
  }
  
  // Watts Voie 1 - Rafraîchi avec unité dynamique
  if (abs(powerWatts1 - lastPowerWatts1) > 0.000001) {
    float pow = 0;
    String currentUnit = "";
    
    tft.fillRect(85, 100, 69, 15, COLOR_BG);
    tft.setTextColor(COLOR_CH1);
    tft.setTextSize(2);

    if (displayLinearUnits && powerWatts1 >= 0) {
      if (powerWatts1 >= 1) {
        pow = powerWatts1;
        currentUnit = "W)";
        if (powerWatts1>=100) tft.setCursor(85,100);
        else if((10.0<=powerWatts1) && (powerWatts1<100.0)) tft.setCursor(97,100);
        else tft.setCursor(109,100);
        tft.print(pow,1);
      } else if (powerWatts1 >= 0.001) {
        pow = powerWatts1 * 1000.0;
        currentUnit = "mW)";
        if (pow>=100) tft.setCursor(85,100);
        else if((10.0<=pow) && (pow<100.0)) tft.setCursor(97,100);
        else tft.setCursor(109,100);
        tft.print(pow,1);
      } else if (powerWatts1 >= 0.000001) {
        pow = powerWatts1 * 1000000.0;
        currentUnit = "uW)";
        if (pow>=100) tft.setCursor(85,100);
        else if((10.0<=pow) && (pow<100.0)) tft.setCursor(97,100);
        else tft.setCursor(109,100);
        tft.print(pow,1);
      } else {
        pow = powerWatts1 * 1000000000.0;
        currentUnit = "nW)";
        if (pow>=100) tft.setCursor(85,100);
        else if((10.0<=pow) && (pow<100.0)) tft.setCursor(97,100);
        else tft.setCursor(109,100);
        tft.print(pow,1);
      }
    } else {
      tft.setCursor(85, 100);
      tft.print(F("---.--"));
      currentUnit = "";
    }
    
    // Mettre à jour l'unité si elle a changé
    if (currentUnit != lastUnit1) {
      tft.fillRect(34, 100, 42, 15, COLOR_BG);
      tft.setTextSize(2);
      tft.setTextColor(COLOR_CH1);
      tft.setCursor(34, 100);
      tft.print(currentUnit);
      lastUnit1 = currentUnit;
    }
    
    lastPowerWatts1 = powerWatts1;
  }
  
  // Watts Voie 2 - Rafraîchi avec unité dynamique
  if (abs(powerWatts2 - lastPowerWatts2) > 0.000001) {
    float pow = 0;
    String currentUnit = "";
    
    tft.fillRect(246, 100, 68, 15, COLOR_BG);
    tft.setTextColor(COLOR_HDR);
    tft.setTextSize(2);

    if (displayLinearUnits && powerWatts2 >= 0) {
      if (powerWatts2 >= 1) {
        pow = powerWatts2;
        currentUnit = "W)";
        if (powerWatts2>=100) tft.setCursor(246,100);
        else if((10.0<=powerWatts2) && (powerWatts2<100.0)) tft.setCursor(258,100);
        else tft.setCursor(269,100);
        tft.print(pow, 1); 
      } else if (powerWatts2 >= 0.001) {
        pow = powerWatts2 * 1000.0;
        currentUnit = "mW)";
        if (pow>=100) tft.setCursor(246,100);
        else if((10.0<=pow) && (pow<100.0)) tft.setCursor(258,100);
        else tft.setCursor(269,100);
        tft.print(pow, 1);
      } else if (powerWatts2 >= 0.000001) {
        pow = powerWatts2 * 1000000.0;
        currentUnit = "uW)";
        if (pow>=100) tft.setCursor(246,100);
        else if((10.0<=pow) && (pow<100.0)) tft.setCursor(258,100);
        else tft.setCursor(269,100);
        tft.print(pow, 1);
      } else {
        pow = powerWatts2 * 1000000000.0;
        currentUnit = "nW)";
        if (pow>=100) tft.setCursor(246,100);
        else if((10.0<=powerWatts2) && (powerWatts2<100.0)) tft.setCursor(258,100);
        else tft.setCursor(269,100);
        tft.print(pow, 1);
      }
    } else {
      tft.setCursor(246, 100);
      tft.print(F("---.--"));
      currentUnit = "";
    }
    
    // Mettre à jour l'unité si elle a changé
    if (currentUnit != lastUnit2) {
      tft.fillRect(194, 100, 42, 15, COLOR_BG);
      tft.setTextSize(2);
      tft.setTextColor(COLOR_HDR);
      tft.setCursor(194, 100);
      tft.print(currentUnit);
      lastUnit2 = currentUnit;
    }
    
    lastPowerWatts2 = powerWatts2;
  }
}

void updateGainValue(float power1, float power2) {
  float currentGainDb = power2 - power1;
  
  // Rafraîchir UNIQUEMENT la VALEUR numérique du gain
  if (abs(currentGainDb - lastGainDb) > 0.1) {
    // Effacer UNIQUEMENT la zone de la valeur (210,208 à 270,223)
    tft.fillRect(234, 208, 60, 15, COLOR_BG);
    tft.setTextSize(2);  
    tft.setTextColor(COLOR_VAL);
    tft.setCursor(234, 208);
    if (currentGainDb >= 0) tft.print("+");
    tft.print(currentGainDb, 1);
    lastGainDb = currentGainDb;
  }
}

void updateControlParameters() {
  updateFreqDisplay();
  updateStepDisplay();
  updateAttDisplay();
}

void updateAllDisplay() {
  updateFreqDisplay();
  updateStepDisplay();
  updateAttDisplay();
}

void updateFreqDisplay() {
  if (abs(currentFreq - lastFreq) > 0.001 || lastFreq == -1.0) {
    tft.fillRect(112, 35, 80, 15, COLOR_BG);
    tft.setTextSize(2);
    tft.setTextColor((currentMode == MODE_FREQUENCY) ? COLOR_VAL : COLOR_HDR);
    tft.setCursor(112, 35);
    tft.print(currentFreq, 1);
    lastFreq = currentFreq;
  }
}

void updateStepDisplay() {
  float currentStep = freqSteps[currentStepIndex];
  
  if (abs(currentStep - lastStep) > 0.001 || lastStep == -1.0) {
    tft.fillRect(274, 40, 36, 10, COLOR_BG);
    tft.setTextSize(1);
    tft.setTextColor((currentMode == MODE_FREQ_STEP) ? COLOR_VAL : COLOR_HDR);
    tft.setCursor(274, 40);
    if (currentStep >= 100) tft.print(currentStep, 0);
    else if (currentStep >= 10) { tft.print(" "); tft.print(currentStep, 0); }
    else { tft.print("  "); tft.print(currentStep, 1); }
    lastStep = currentStep;
  }
}

void updateAttDisplay() {
  if (abs(currentAtt1 - lastAtt) > 0.001 || lastAtt == -1.0) {
    tft.fillRect(85, 150, 65, 15, COLOR_BG);
    tft.setTextSize(2);
    tft.setTextColor((currentMode == MODE_ATT_SONDE1) ? COLOR_VAL : COLOR_CH1);
    if (currentAtt1>9.99) tft.setCursor(97, 150);
    else tft.setCursor(109, 150);
    tft.print(currentAtt1, 1);
    lastAtt = currentAtt1;
  }
  
  if (abs(currentAtt2 - lastAtt2) > 0.001 || lastAtt2 == -1.0) {
    tft.fillRect(258, 150, 50, 15, COLOR_BG);
    tft.setTextSize(2);
    tft.setTextColor((currentMode == MODE_ATT_SONDE2) ? COLOR_VAL : COLOR_HDR);
    if (currentAtt2>9.9) tft.setCursor(258,150);
    else tft.setCursor(269, 150);
    tft.print(currentAtt2, 1);
    lastAtt2 = currentAtt2;
  }
}

void updateMessage(const char* message) {
  // Rafraîchir UNIQUEMENT la partie MODE (pas les labels constants)
  if (String(message) != lastMessage) {
    // EFFACER UNIQUEMENT la zone du mode (82,205 à 182,215)
    tft.fillRect(82, 208, 70, 10, COLOR_BG);
    tft.setTextSize(1);
    tft.setTextColor(COLOR_HDR);
    tft.setCursor(82, 210);
    tft.print(message);
    lastMessage = message;
  }
}
