/*
 * RF_PowerMeter.h - Library for RF Power Measurements with AD8307 detector
 * Created by F4GSC (adapted by Claude), May 2025
 * Released into the public domain
 * VERSION CORRIGEE - Fix calibration et interpolation
 */

#ifndef RF_PowerMeter_h
#define RF_PowerMeter_h

#include "Arduino.h"

// Définition des types de sondes
#define SONDE_1 1  // AD8307
#define SONDE_2 2  // AD8307

class RF_PowerMeter
{
  public:
    // Constructeur
    RF_PowerMeter(int analogPin, float vRef);
    
    // Méthodes de configuration

    void setSonde(int sondeType);
    void setFrequency(float freq);
    void setAttenuation(float atten);
    void setVref(float vRef);
    void setDebugMode(bool debug);  // NOUVEAU - Mode debug
    
    // Méthodes de mesure
    float readVoltage();
    float calculatePower();           // Puissance sans atténuation
    float calculateCorrectedPower();  // Puissance avec atténuation
    float convertToWatts(float dbm);  // Conversion dBm vers Watt
    
    // Accesseurs
    float getFrequency();
    float getAttenuation();
    int getSondeType();
    float getVoltage();
    String getSondeName();
    
  private:
    // Variables internes
    int _analogPin;
    float _vRef;
    float _currentFreq;
    float _attenuationDB;
    int _sondeType;
    float _lastVoltage;
    bool _debugMode = false;          // NOUVEAU - Mode debug
    const int _numSamples = 256;       // REDUIT pour plus de rapidité
    
    // Constantes pour les tableaux de calibration
    static const int NB_POINTS = 9;
    static const int NB_POINTS_F = 10;
    static const float Pinc[NB_POINTS];
    static const int FREQ_TABLE[NB_POINTS_F];
    
    // Tableaux de calibration pour SONDE_1
    static const float Vm_1MHz_SONDE1[NB_POINTS];
    static const float Vm_10MHz_SONDE1[NB_POINTS];
    static const float Vm_50MHz_SONDE1[NB_POINTS];
    static const float Vm_100MHz_SONDE1[NB_POINTS];
    static const float Vm_145MHz_SONDE1[NB_POINTS];
    static const float Vm_300MHz_SONDE1[NB_POINTS];
    static const float Vm_435MHz_SONDE1[NB_POINTS];
    static const float Vm_600MHz_SONDE1[NB_POINTS];
    static const float Vm_800MHz_SONDE1[NB_POINTS];
    static const float Vm_1000MHz_SONDE1[NB_POINTS];
    
    // Tableaux de calibration pour SONDE_2
    static const float Vm_1MHz_SONDE2[NB_POINTS];
    static const float Vm_10MHz_SONDE2[NB_POINTS];
    static const float Vm_50MHz_SONDE2[NB_POINTS];
    static const float Vm_100MHz_SONDE2[NB_POINTS];
    static const float Vm_145MHz_SONDE2[NB_POINTS];
    static const float Vm_300MHz_SONDE2[NB_POINTS];
    static const float Vm_435MHz_SONDE2[NB_POINTS];
    static const float Vm_600MHz_SONDE2[NB_POINTS];
    static const float Vm_800MHz_SONDE2[NB_POINTS];
    static const float Vm_1000MHz_SONDE2[NB_POINTS];
  
    // Méthodes internes de calcul
    const float* getVmArrayForFreq(int freq);
    float calculatePowerAtFreq(float voltage, int freq);
    float calculatePowerWithFreqInterpolation(float voltage, float freq);
};

#endif
