/*
 * RF_PowerMeter.cpp - Library for RF Power Measurements with AD8307 detector
 * Created by F4GSC (adapted by Claude), May 2025
 * Released into the public domain
 * VERSION CORRIGEE - Fix calibration et interpolation
 */

#include "Arduino.h"
#include "RF_PowerMeter.h"

// Définir les constantes statiques
const int RF_PowerMeter::FREQ_TABLE[NB_POINTS_F]       = {1,10,50,100,145,300,435,600,800,1000};
const float RF_PowerMeter::Pinc[NB_POINTS]             = {10,     5,        0,   -10,   -30,   -60,   -70,   -75,  -80};

// TABLEAUX CORRIGES - Les tensions décroissent avec la puissance (caractéristique normale AD8307)
const float RF_PowerMeter::Vm_1MHz_SONDE1[NB_POINTS]   = {2.509, 2.385, 2.258, 2.010, 1.492, 0.737, 0.484, 0.368, 0.282};
const float RF_PowerMeter::Vm_10MHz_SONDE1[NB_POINTS]  = {2.493, 2.367, 2.239, 1.990, 1.480, 0.725, 0.470, 0.354, 0.274};
const float RF_PowerMeter::Vm_50MHz_SONDE1[NB_POINTS]  = {2.479, 2.345, 2.215, 1.965, 1.468, 0.723, 0.483, 0.370, 0.280};
const float RF_PowerMeter::Vm_100MHz_SONDE1[NB_POINTS] = {2.472, 2.348, 2.210, 1.956, 1.452, 0.703, 0.454, 0.344, 0.268};
const float RF_PowerMeter::Vm_145MHz_SONDE1[NB_POINTS] = {2.438, 2.326, 2.196, 1.929, 1.431, 0.702, 0.455, 0.334, 0.261};
const float RF_PowerMeter::Vm_300MHz_SONDE1[NB_POINTS] = {2.395, 2.264, 2.135, 1.877, 1.362, 0.586, 0.359, 0.272, 0.233};
const float RF_PowerMeter::Vm_435MHz_SONDE1[NB_POINTS] = {2.320, 2.229, 2.052, 1.806, 1.278, 0.503, 0.287, 0.242, 0.223};
const float RF_PowerMeter::Vm_600MHz_SONDE1[NB_POINTS] = {2.177, 2.104, 1.920, 1.656, 1.103, 0.293, 0.257, 0.257, 0.257};
const float RF_PowerMeter::Vm_800MHz_SONDE1[NB_POINTS] = {1.920, 1.859, 1.698, 1.389, 0.810, 0.248, 0.223, 0.210, 0.210};
const float RF_PowerMeter::Vm_1000MHz_SONDE1[NB_POINTS]= {1.617, 1.568, 1.405, 1.066, 0.477, 0.245, 0.245, 0.245, 0.245};


// Tableaux pour SONDE_2 (corrigés également)
const float RF_PowerMeter::Vm_1MHz_SONDE2[NB_POINTS]    = {2.531, 2.405, 2.256, 2.001, 1.495, 0.735, 0.484, 0.360, 0.185};
const float RF_PowerMeter::Vm_10MHz_SONDE2[NB_POINTS]   = {2.512, 2.386, 2.254, 2.001, 1.482, 0.713, 0.461, 0.342, 0.256};
const float RF_PowerMeter::Vm_50MHz_SONDE2[NB_POINTS]   = {2.496, 2.363, 2.227, 1.972, 1.471, 0.715, 0.461, 0.343, 0.250};
const float RF_PowerMeter::Vm_100MHz_SONDE2[NB_POINTS]  = {2.484, 2.363, 2.221, 1.962, 1.450, 0.696, 0.442, 0.328, 0.249};
const float RF_PowerMeter::Vm_145MHz_SONDE2[NB_POINTS]  = {2.445, 2.338, 2.206, 1.933, 1.428, 0.686, 0.438, 0.318, 0.240};
const float RF_PowerMeter::Vm_300MHz_SONDE2[NB_POINTS]  = {2.404, 2.262, 2.136, 1.877, 1.353, 0.580, 0.399, 0.318, 0.266};
const float RF_PowerMeter::Vm_435MHz_SONDE2[NB_POINTS]  = {2.318, 2.229, 2.040, 1.795, 1.264, 0.476, 0.262, 0.219, 0.200};
const float RF_PowerMeter::Vm_600MHz_SONDE2[NB_POINTS]  = {2.170, 2.100, 1.907, 1.639, 1.082, 0.266, 0.266, 0.266, 0.266};
const float RF_PowerMeter::Vm_800MHz_SONDE2[NB_POINTS]  = {1.916, 1.851, 1.680, 1.366, 0.793, 0.193, 0.193, 0.193, 0.193};
const float RF_PowerMeter::Vm_1000MHz_SONDE2[NB_POINTS] = {1.607, 1.562, 1.342, 1.041, 0.478, 0.266, 0.266, 0.266, 0.266};


// Constructeur
RF_PowerMeter::RF_PowerMeter(int analogPin, float vRef) 
 {
  _analogPin = analogPin;
  _vRef = vRef;
  _currentFreq = 145.0;    // Fréquence par défaut
  _attenuationDB = 0.0;    // Atténuation par défaut
  _sondeType = SONDE_1;    // Type de sonde par défaut
  _lastVoltage = 0.0;
 }

// Methodes de configuration
void RF_PowerMeter::setSonde(int sondeType) 
 {
  if (sondeType == SONDE_1 || sondeType == SONDE_2) _sondeType = sondeType;
 }

void RF_PowerMeter::setFrequency(float freq) {
  if (freq >= 1.0 && freq <= 1300.0) {  // Étendu jusqu'à 1300MHz
    _currentFreq = freq;
  }
}

void RF_PowerMeter::setAttenuation(float atten) {
  if (atten >= 0.0 && atten <= 60.0) {
    _attenuationDB = atten;
  }
}

void RF_PowerMeter::setVref(float vRef) {
  if (vRef > 0.0) {
    _vRef = vRef;
  }
}

// Methodes de mesure
float RF_PowerMeter::readVoltage() {
  long sum = 0;
  
  // Faire la moyenne de plusieurs lectures
  for (int i = 0; i < _numSamples; i++) 
   {
     sum += analogRead(_analogPin);
     delayMicroseconds(100);  // Délai plus court pour des mesures plus rapides
    }
  
  // Calculer la moyenne
  float average = (float)sum / _numSamples;
  
  // Convertir la valeur analogique en tension
  _lastVoltage = average * _vRef / 1023.0;
  
  return _lastVoltage;
}

float RF_PowerMeter::calculatePower() {
  // Utilise la dernière tension mesuree ou effectue une nouvelle mesure
  float voltage = _lastVoltage > 0.0 ? _lastVoltage : readVoltage();
  
  // Appliquer l'interpolation de fréquence
  return calculatePowerWithFreqInterpolation(voltage, _currentFreq);
}

float RF_PowerMeter::calculateCorrectedPower() {
  // Calculer la puissance et ajouter l'atténuation
  return calculatePower() + _attenuationDB;
}

float RF_PowerMeter::convertToWatts(float dbm) {
  // P(W) = 10^((dBm - 30) / 10)
  return pow(10, (dbm - 30) / 10.0);
}

// Accesseurs
float RF_PowerMeter::getFrequency() {
  return _currentFreq;
}

float RF_PowerMeter::getAttenuation() {
  return _attenuationDB;
}

int RF_PowerMeter::getSondeType() {
  return _sondeType;
}

float RF_PowerMeter::getVoltage() {
  return _lastVoltage;
}

String RF_PowerMeter::getSondeName() {
  return "AD8307";
}

// Méthodes internes
const float* RF_PowerMeter::getVmArrayForFreq(int freq) {
  if (_sondeType == SONDE_1) {
    switch (freq) {
      case 1: return Vm_1MHz_SONDE1;
      case 10: return Vm_10MHz_SONDE1;
      case 50: return Vm_50MHz_SONDE1;
      case 100: return Vm_100MHz_SONDE1;
      case 145: return Vm_145MHz_SONDE1;
      case 300: return Vm_300MHz_SONDE1;
      case 435: return Vm_435MHz_SONDE1;
      case 600: return Vm_600MHz_SONDE1;
      case 800: return Vm_800MHz_SONDE1;
      case 1000: return Vm_1000MHz_SONDE1;
      default: return NULL;
    }
  } else { // SONDE_2
    switch (freq) {
      case 1: return Vm_1MHz_SONDE2;
      case 10: return Vm_10MHz_SONDE2;
      case 50: return Vm_50MHz_SONDE2;
      case 100: return Vm_100MHz_SONDE2;
      case 145: return Vm_145MHz_SONDE2;
      case 300: return Vm_300MHz_SONDE2;
      case 435: return Vm_435MHz_SONDE2;
      case 600: return Vm_600MHz_SONDE2;
      case 800: return Vm_800MHz_SONDE2;
      case 1000: return Vm_1000MHz_SONDE2;
      default: return NULL;
    }
  }
}

float RF_PowerMeter::calculatePowerAtFreq(float voltage, int freq) 
{
  const float* Vm_array = getVmArrayForFreq(freq);
  
  if (Vm_array == NULL) 
    {
    return -999.9; // Erreur: fréquence non reconnue
   }
  
  // CORRECTION CRITIQUE: Vérifier si la tension est dans la plage valide
  // Si tension > tension max (correspondant à +20dBm)
  if (voltage >= Vm_array[0]) 
     {
        return Pinc[0]; // +20dBm
     }
  
  // Si tension < tension min (correspondant à -60dBm)  
  if (voltage <= Vm_array[NB_POINTS - 1]) 
     {
    	return Pinc[NB_POINTS - 1]; // -60dBm
     }
  
  // Recherche de l'intervalle pour l'interpolation
  // ATTENTION: Dans le tableau AD8307, les tensions DÉCROISSENT avec la puissance
  // Donc: haute tension = haute puissance, basse tension = basse puissance
  
  for (int i = 0; i < NB_POINTS - 1; i++) {
    // Vérifier si la tension est entre Vm_array[i] et Vm_array[i+1]
    if (voltage <= Vm_array[i] && voltage >= Vm_array[i + 1]) 
 	{     
      	// Interpolation linéaire corrigée
      	float v1 = Vm_array[i];     // Tension point 1
      	float v2 = Vm_array[i + 1]; // Tension point 2
      	float p1 = Pinc[i];         // Puissance point 1
      	float p2 = Pinc[i + 1];     // Puissance point 2
      
     	// Éviter la division par zéro
      	if (abs(v2 - v1) < 0.001) 
		{
       	 	Serial.println(F("Tensions identiques, retour moyenne"));
        	return (p1 + p2) / 2.0;
      		}
      
      // Formule d'interpolation linéaire
      float power = p1 + (voltage - v1) * (p2 - p1) / (v2 - v1);
 
      
      return power;
    }
  }
  
  return -999.9;
}

float RF_PowerMeter::calculatePowerWithFreqInterpolation(float voltage, float freq) 
{
  // Si la fréquence correspond à l'une des fréquences précalibrées, pas besoin d'interpolation
  for (int i = 0; i < NB_POINTS_F; i++) 
     {
       if (abs(freq - FREQ_TABLE[i]) < 0.001) 
       {
         return calculatePowerAtFreq(voltage, FREQ_TABLE[i]);
       }
   
     }
 
  // Trouver les deux fréquences les plus proches entourant la fréquence cible
  int idxLow = -1, idxHigh = -1;
  
  // Cas spéciaux: fréquences hors limites
  if (freq <= FREQ_TABLE[0]) {
    return calculatePowerAtFreq(voltage, FREQ_TABLE[0]);
  }
  if (freq >= FREQ_TABLE[NB_POINTS_F - 1]) {
    return calculatePowerAtFreq(voltage, FREQ_TABLE[NB_POINTS_F - 1]);
  }
  
  // Recherche de l'intervalle
  for (int i = 0; i < NB_POINTS_F - 1; i++) 
  {
    if (FREQ_TABLE[i] <= freq && freq <= FREQ_TABLE[i + 1]) 
     {
      idxLow = i;
      idxHigh = i + 1;
      break;
    }
  }
  
  if (idxLow == -1 || idxHigh == -1) 
    {
    return -999.9; // Fréquence hors plage
    }

  int freqLow = FREQ_TABLE[idxLow];
  int freqHigh = FREQ_TABLE[idxHigh];

  float powerLow = calculatePowerAtFreq(voltage, freqLow);
  float powerHigh = calculatePowerAtFreq(voltage, freqHigh);

  if (powerLow == -999.9 || powerHigh == -999.9) 
    {
      return -999.9;
    }
 
  // Interpolation linéaire entre les deux puissances
  float interpolatedPower = powerLow + (freq - freqLow) * (powerHigh - powerLow) / (freqHigh - freqLow);
  return interpolatedPower;
}

// Méthode pour activer/désactiver le debug
void RF_PowerMeter::setDebugMode(bool debug) 
{
  _debugMode = debug;
}